""" Scripts containing functions related to parsing MPT model files.

| Copyright 2017 Cognitive Computation Lab
| University of Freiburg
| Nicolas Riesterer <riestern@tf.uni-freiburg.de>

"""

import re


def is_number(val):
    """ Checks, if the given argument is a number.

    Parameters
    ----------
    val : str
        String to check.

    Returns
    -------
    Boolean
        True, if val is a valid number representation, false otherwise.

    """

    try:
        float(val)
    except ValueError:
        return False
    return True

def parse(model_filepath):
    """ Parses a model file to obtain the formulae for individual outcome
    categories and subtrees.

    Parameters
    ----------
    model_filepath : str
        File to parse the model from (MPTinR format).

    Returns
    -------
    subtrees : list(list(str))
        List structures containing subtrees consisting of lists of outcome
        category formulae.

    params : list(str)
        List of parameter identifiers.

    """

    with open(model_filepath) as model_file:
        params = []
        subtrees = []
        current_subtree = []
        for line in model_file.readlines():
            line = line.strip()
            if line.startswith('#'):
                continue

            if not line and current_subtree:
                subtrees.append(current_subtree)
                current_subtree = []
                continue

            # Normalize the line by stripping it of uninsteresting whitespace
            # and removing commented information.
            comment_start = line.find('#')
            if comment_start > 0:
                line = line[:line.find('#')].strip()

            if not line:
                continue

            # Add the formula to the current subtree and extend the parameter
            # list.
            current_subtree.append(line)
            cur_params = set([x for x in re.split(r'\W', line)
                              if len(x) > 0 and not is_number(x)])
            params.extend([x for x in cur_params if x not in params])

        # Manually append the last subtree and return the result
        if current_subtree:
            subtrees.append(current_subtree)
        return subtrees, sorted(params)
