"""
    Copyright 2017 Cognitive Computation Lab
    University of Freiburg
    Nicolas Riesterer <riestern@tf.uni-freiburg.de>

    Implementation for the feature selection model of the atmosphere theory as
    defined by [Revlis1975].

    [Revlis1975] Revlis, Russell. "Two models of syllogistic reasoning: Feature
        selection and conversion." Journal of Verbal Learning and Verbal
        Behavior 14.2 (1975): 180-195.
"""

import random

from modelbase import CognitiveModel

def encode_premise(premise):
    """
    Construct the model's representation for premises.

    Args:
        premise: Problem premise in character representation (e.g. 'A').

    Returns:
        Representation of the premise given as a tuple according to
        the table reported in [Revlis1975]:

        A   All A are B         [1,1]
        E   No A are B          [1,0]
        I   Some A are B        [0,1]
        O   Some A are not B    [0,0]

    Raises:
        ValueError: An error indicating invalid premise identifier.
    """

    if premise == "A":
        return (1, 1)
    if premise == "E":
        return (1, 0)
    if premise == "I":
        return (0, 1)
    if premise == "O":
        return (0, 0)

    raise ValueError(
        "Premise identifier not in [A,E,I,O]: {}".format(premise))

def composite_representation(p_1, p_2):
    """
    Constructs the composite representation from the two premise encodings
    based on the two rules defined in [Revlis1975]. In summary, the rules
    define the composite representation to be equal to the conjunction of
    both premise representations.

    Args:
        p_1: Representation for the first premise (e.g. (0, 1))
        p_2: Representation for the second premise (e.g. (1, 0))

    Returns:
        Composite representation given as a tuple of values that are either
        0 or 1 (e.g. (0, 1)).

    >>> fs = FeatureSelection()
    >>> fs.composite_representation([1, 1], [1, 1])
    (1, 1)
    >>> fs.composite_representation([1, 0], [1, 1])
    (1, 0)
    >>> fs.composite_representation([1, 1], [0, 1])
    (0, 1)
    >>> fs.composite_representation([0, 0], [0, 0])
    (0, 0)
    """

    return (p_1[0] & p_2[0], p_1[1] & p_2[1])

def create_candidate_conclusions():
    """
    Constructs the list of candidate conclusions (e.g. Aac, Aca, Eac, ...)
    along with their corresponding atmosphere encodings.

    Returns:
        Tuple containing lists of conclusions and corresponding
        atmosphere encodings.
    """

    moods = ["A", "E", "I", "O"]
    conclusions = []
    conclusion_reps = []
    for mood in moods:
        for direction in ["ac", "ca"]:
            conclusions.append(mood + direction)
            conclusion_reps.append(encode_premise(mood))

    return conclusions, conclusion_reps

class FeatureSelection(CognitiveModel):
    """
    Feature Selection model of formal reasoning as reported by [Revlis1975].
    """

    def predict(self, task):
        """
        Predicts the set of accepted results as specified in [Revlis1975].
        Accounts for the missing information about direction by uniformly
        drawing the direction for the predicted quantifier.

        Args:
            task: String representation of the task to generate a prediction
                for (e.g. 'AA1').

        Returns:
            A string representation of the answer (e.g. 'Aac').
        """

        # Extract the premise information
        p_1 = task[0]
        p_2 = task[1]

        # Extract representation for premises
        rep1 = encode_premise(p_1)
        rep2 = encode_premise(p_2)

        # Create the composite representation
        comp = composite_representation(rep1, rep2)

        # Construct the set of possible conclusions
        conclusions, conclusion_reps = create_candidate_conclusions()

        # Compare the conclusions with the composite representation
        pred = []
        for idx, conclusion in enumerate(conclusions):
            cur_rep = conclusion_reps[idx]

            if cur_rep == comp:
                pred.append(conclusion)

        # Return the set of predictions
        return random.choice(pred)
